//	TorusGamesMatteView.m
//
//	Lays out itsMessageView, itsGameEnclosureView and itsKeyboardView
//	appropriately for portrait and landscape orientations on iPad.
//
//	© 2021 by Jeff Weeks
//	See TermsOfUse.txt

#import "TorusGamesMatteView.h"

#define GAME_ENCLOSURE_MARGIN_MIN	 10.0
#define GAME_ENCLOSURE_MARGIN_MAX	 16.0
#define KEYBOARD_HEIGHT_LANDSCAPE	160.0
#define MIN_SIDE_PANEL_WIDTH		292.0	//	width of Torus Games Crossword keyboard
											//	on 320×568 display in landscape orientation


typedef struct
{
	CGRect	itsGameEnclosureFrame,
			itsMessageFrame,
			itsKeyboardFrame;
} TorusGamesLayout;


//	Privately-declared properties and methods
@interface TorusGamesMatteView()
- (TorusGamesLayout)portraitLayoutInRect:(CGRect)aVisibleRect;
- (TorusGamesLayout)landscapeLayoutInRect:(CGRect)aVisibleRect;
- (CGFloat)preferredGameEnclosureSizeInRect:(CGRect)aVisibleRect;
- (CGFloat)gameEnclosureMarginInRect:(CGRect)aVisibleRect;
@end


@implementation TorusGamesMatteView
{
	id<TorusGamesMatteViewDelegate>	__weak	itsDelegate;
	
	GameType								itsGame;

	UIView									*itsMessageView,
											*itsGameEnclosureView,
											*itsKeyboardView;
	
	CGFloat									itsTopMargin,		//	height covered by transparent status bar
											itsBottomMargin;	//	height covered by toolbar
}


- (id)initWithDelegate:(id<TorusGamesMatteViewDelegate>)aDelegate frame:(CGRect)aFrame game:(GameType)aGame
{
	self = [super initWithFrame:aFrame];
	if (self != nil)
	{
		itsDelegate		= aDelegate;
		itsGame			= aGame;
		itsTopMargin	= 0.0;	//	to be overwritten later
		itsBottomMargin	= 0.0;	//	to be overwritten later
	}
	return self;
}


- (void)addMessageView:(UIView *)aMessageView
{
	itsMessageView = aMessageView;
	[self addSubview:aMessageView];
}

- (void)addGameEnclosureView:(UIView *)aGameEnclosureView
{
	itsGameEnclosureView = aGameEnclosureView;
	[self addSubview:aGameEnclosureView];
}

- (void)addKeyboardView:(UIView *)aKeyboardView
{
	itsKeyboardView = aKeyboardView;
	[self addSubview:aKeyboardView];
}


- (void)setMarginTop:(CGFloat)aTopMargin bottom:(CGFloat)aBottomMargin
{
	itsTopMargin	= aTopMargin;
	itsBottomMargin	= aBottomMargin;
}

- (void)layoutSubviews
{
	CGRect				theVisibleRect;
	TorusGamesLayout	thePortraitLayout,
						theLandscapeLayout,
						*theLayout;

	theVisibleRect	= (CGRect)
						{
							{
								[self bounds].origin.x,
								[self bounds].origin.y + itsTopMargin},
							{
								[self bounds].size.width,
								[self bounds].size.height - (itsTopMargin + itsBottomMargin)
							}
						};
	
	thePortraitLayout	= [self  portraitLayoutInRect:theVisibleRect];
	theLandscapeLayout	= [self landscapeLayoutInRect:theVisibleRect];

	//	Choose the portrait or landscape layout based not on a strict comparison
	//	of theVisibleRect's width and height, but rather on which layout
	//	would provide a larger area for itsGameEnclosureView.
	//	Note:  The game enclosure is always square,
	//	so it makes no difference whether we compares widths or heights.
	//
	if (thePortraitLayout.itsGameEnclosureFrame.size.width
	 > theLandscapeLayout.itsGameEnclosureFrame.size.width)
	{
		theLayout = &thePortraitLayout;

		[itsDelegate matteViewDidSetLayoutOrientation:MatteViewPortraitLayout];
	}
	else
	{
		theLayout = &theLandscapeLayout;

		[itsDelegate matteViewDidSetLayoutOrientation:MatteViewLandscapeLayout];
	}
	
	[itsGameEnclosureView	setFrame:theLayout->itsGameEnclosureFrame	];
	[itsMessageView			setFrame:theLayout->itsMessageFrame			];
	[itsKeyboardView		setFrame:theLayout->itsKeyboardFrame		];
}

- (TorusGamesLayout)portraitLayoutInRect:(CGRect)aVisibleRect
{
	CGFloat				thePreferredGameEnclosureSize;
	CGFloat				theHeaderSpace,
						theFooterSpace,
						theHorizontalInset;
	TorusGamesLayout	thePortraitLayout;
	
	thePreferredGameEnclosureSize = [self preferredGameEnclosureSizeInRect:aVisibleRect];

	switch (itsGame)
	{
		case Game2DCrossword:
			theHeaderSpace	= ceil(0.5 * (aVisibleRect.size.height - thePreferredGameEnclosureSize));
			theFooterSpace	= ceil(0.5 * (aVisibleRect.size.height - thePreferredGameEnclosureSize));
			//	The Crossword wants at least 96pt to display
			//	the keyboard at the bottom of the window...
			if (theFooterSpace < 96.0)
				theFooterSpace = 96.0;
			//	...and at least 40pt to display the clue
			//	at the top of the window.
			if (theHeaderSpace < 40.0)
				theHeaderSpace = 40.0;
			break;
		
		case Game2DWordSearch:
#ifdef GAME_CONTENT_FOR_SCREENSHOT
			//	Let's allow the same header and footer space in Word Search
			//	as in the other games, so that the square game boards
			//	will all line up nicely when the screenshots are viewed
			//	in the App Store.
			//
			theHeaderSpace	= ceil(0.5 * (aVisibleRect.size.height - thePreferredGameEnclosureSize));
			theFooterSpace	= ceil(0.5 * (aVisibleRect.size.height - thePreferredGameEnclosureSize));
#else
			theHeaderSpace	= ceil(0.625 * (aVisibleRect.size.height - thePreferredGameEnclosureSize));
			theFooterSpace	= ceil(0.375 * (aVisibleRect.size.height - thePreferredGameEnclosureSize));
#endif
			//	The Word Search wants at least 96pt to display
			//	the word list at the top of the window.
			if (theHeaderSpace < 96.0)
				theHeaderSpace = 96.0;
			break;
		
		case Game2DPool:
			theHeaderSpace	= ceil(0.5 * (aVisibleRect.size.height - thePreferredGameEnclosureSize));
			theFooterSpace	= ceil(0.5 * (aVisibleRect.size.height - thePreferredGameEnclosureSize));
			//	The Pool game wants at least 64pt to display
			//	the status message at the top of the window.
			if (theHeaderSpace < 64.0)
				theHeaderSpace = 64.0;
			break;
		
		case Game3DMaze:
			theHeaderSpace	= ceil(0.5 * (aVisibleRect.size.height - thePreferredGameEnclosureSize));
			theFooterSpace	= ceil(0.5 * (aVisibleRect.size.height - thePreferredGameEnclosureSize));
			if ([itsMessageView isKindOfClass:[UILabel class]]
			 && [((UILabel *)itsMessageView) text] != nil)
			{
				if (theHeaderSpace < 64.0)
					theHeaderSpace = 64.0;
			}
			break;
		
		default:
			theHeaderSpace	= ceil(0.5 * (aVisibleRect.size.height - thePreferredGameEnclosureSize));
			theFooterSpace	= ceil(0.5 * (aVisibleRect.size.height - thePreferredGameEnclosureSize));
			break;
	}
	
	//	Recompute thePreferredGameEnclosureSize allowing
	//	for the possible rounding-up in ceil()
	//	and the possible adjustments for the Crossword and Word Search.
	thePreferredGameEnclosureSize = aVisibleRect.size.height - (theHeaderSpace + theFooterSpace);

	theHorizontalInset = ceil(0.5 * (aVisibleRect.size.width  - thePreferredGameEnclosureSize));

	thePortraitLayout.itsGameEnclosureFrame = (CGRect)
		{
			{
				aVisibleRect.origin.x + theHorizontalInset,
				aVisibleRect.origin.y + theHeaderSpace
			},
			{
				thePreferredGameEnclosureSize,
				thePreferredGameEnclosureSize
			}
		};
	thePortraitLayout.itsMessageFrame = (CGRect)
		{
			{
				aVisibleRect.origin.x + theHorizontalInset,
				aVisibleRect.origin.y + 0.0
			},
			{
				aVisibleRect.size.width - 2.0*theHorizontalInset,
				theHeaderSpace
			}
		};
	thePortraitLayout.itsKeyboardFrame = (CGRect)
		{
			{
				aVisibleRect.origin.x + theHorizontalInset,
				aVisibleRect.origin.y + (aVisibleRect.size.height - theFooterSpace)
			},
			{
				aVisibleRect.size.width - 2.0*theHorizontalInset,
				theFooterSpace
			}
		};
	
	return thePortraitLayout;
}

- (TorusGamesLayout)landscapeLayoutInRect:(CGRect)aVisibleRect
{
	CGFloat				thePreferredGameEnclosureSize,
						theGamePanelVerticalInset,
						theSidePanelVerticalInset,
						theHorizontalInset,
						theGamePanelWidth,
						theSidePanelWidth;
	TorusGamesLayout	theLandscapeLayout;

	thePreferredGameEnclosureSize = [self preferredGameEnclosureSizeInRect:aVisibleRect];

	if (itsGame == Game2DCrossword
	 || itsGame == Game2DWordSearch
	 || itsGame == Game2DPool
	 || itsGame == Game2DApples
	 || (	itsGame == Game3DMaze
		 && [itsMessageView isKindOfClass:[UILabel class]]
		 && [[((UILabel *)itsMessageView) text] length] > 0))
	{
		//	Allow room for the side panel.

		theGamePanelVerticalInset	= floor(0.5 * (aVisibleRect.size.height - thePreferredGameEnclosureSize));
		theSidePanelVerticalInset	= [self gameEnclosureMarginInRect:aVisibleRect];
		theHorizontalInset			= [self gameEnclosureMarginInRect:aVisibleRect];
		
		theGamePanelWidth			= theHorizontalInset + thePreferredGameEnclosureSize + theHorizontalInset;
		theSidePanelWidth			= aVisibleRect.size.width - theGamePanelWidth;
		
		//	Insist that the side panel have width at least MIN_SIDE_PANEL_WIDTH
		//	to ensure that the keyboard be usable.
		if (theSidePanelWidth < MIN_SIDE_PANEL_WIDTH)
		{
			theSidePanelWidth				= MIN_SIDE_PANEL_WIDTH;
			theGamePanelWidth				= aVisibleRect.size.width - theSidePanelWidth;
			thePreferredGameEnclosureSize	= theGamePanelWidth - 2.0*theHorizontalInset;
			theGamePanelVerticalInset		= floor(0.5 * (aVisibleRect.size.height - thePreferredGameEnclosureSize));
		}

		theLandscapeLayout.itsGameEnclosureFrame = (CGRect)
			{
				{
					aVisibleRect.origin.x + theHorizontalInset,
					aVisibleRect.origin.y + theGamePanelVerticalInset
				},
				{
					thePreferredGameEnclosureSize,
					thePreferredGameEnclosureSize
				}
			};
		theLandscapeLayout.itsMessageFrame = (CGRect)
			{
				{
					aVisibleRect.origin.x + theGamePanelWidth,
					aVisibleRect.origin.y + theSidePanelVerticalInset
				},
				{
					theSidePanelWidth - theHorizontalInset,
					0.5 * aVisibleRect.size.height  -  theSidePanelVerticalInset
				}
			};
		theLandscapeLayout.itsKeyboardFrame = (CGRect)
			{
				{
					aVisibleRect.origin.x + theGamePanelWidth,
					aVisibleRect.origin.y + aVisibleRect.size.height - theSidePanelVerticalInset - KEYBOARD_HEIGHT_LANDSCAPE
				},
				{
					theSidePanelWidth - theHorizontalInset,
					KEYBOARD_HEIGHT_LANDSCAPE
				}
			};
	}
	else
	{
		//	Center the board, with no visible side panel.

		theGamePanelVerticalInset	= floor(0.5 * (aVisibleRect.size.height - thePreferredGameEnclosureSize));
		theHorizontalInset			= floor(0.5 * (aVisibleRect.size.width  - thePreferredGameEnclosureSize));

		theLandscapeLayout.itsGameEnclosureFrame = (CGRect)
			{
				{
					aVisibleRect.origin.x + theHorizontalInset,
					aVisibleRect.origin.y + theGamePanelVerticalInset
				},
				{
					thePreferredGameEnclosureSize,
					thePreferredGameEnclosureSize
				}
			};
		theLandscapeLayout.itsMessageFrame	= (CGRect){{-1.0, -1.0}, {1.0, 1.0}};
		theLandscapeLayout.itsKeyboardFrame	= (CGRect){{-1.0, -1.0}, {1.0, 1.0}};
	}
	
	return theLandscapeLayout;
}

- (CGFloat)preferredGameEnclosureSizeInRect:(CGRect)aVisibleRect
{
	CGFloat	theAvailableSize,
			theGameEnclosureMargin,
			thePreferredGameEnclosureSize;

	theAvailableSize				= MIN(aVisibleRect.size.width, aVisibleRect.size.height);
	theGameEnclosureMargin			= [self gameEnclosureMarginInRect:aVisibleRect];
	thePreferredGameEnclosureSize	= theAvailableSize - 2.0*theGameEnclosureMargin;
	
	return thePreferredGameEnclosureSize;
}

- (CGFloat)gameEnclosureMarginInRect:(CGRect)aVisibleRect
{
	CGFloat	theAvailableSize,
			theGameEnclosureMargin;

	theAvailableSize = MIN(aVisibleRect.size.width, aVisibleRect.size.height);

	theGameEnclosureMargin = floor(0.02 * theAvailableSize);

	if (theGameEnclosureMargin < GAME_ENCLOSURE_MARGIN_MIN)
		theGameEnclosureMargin = GAME_ENCLOSURE_MARGIN_MIN;

	if (theGameEnclosureMargin > GAME_ENCLOSURE_MARGIN_MAX)
		theGameEnclosureMargin = GAME_ENCLOSURE_MARGIN_MAX;
	
	return theGameEnclosureMargin;
}

- (void)refreshLayoutForGame:(GameType)aGame
{
	if (itsGame != aGame)
	{
		itsGame = aGame;

		if ([[self traitCollection] horizontalSizeClass] == UIUserInterfaceSizeClassRegular)
		{
			//	The Change Game popover leaves the various subviews visible,
			//	so let's animate them to their new positions.
			[UIView animateWithDuration:0.5 animations:
			^{
				//	No need to pass "weak self" here,
				//	because this code gets run immediately.
				[self setNeedsLayout];
				[self layoutIfNeeded];
			}];
		}
		else
		{
			//	The Change Game view covers the whole screen,
			//	so let's move the various subviews to their new positions immediately,
			//	so they'll be in final position as the Change Game view slides away.
			[self setNeedsLayout];
			[self layoutIfNeeded];
		}
	}
}


@end
